<?php

namespace App\Services;

use App\DTOs\Auth\UpdatePasswordDTO;
use App\Models\Role;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Log;
use App\DTOs\Auth\RegisterDTO;
use Illuminate\Support\Facades\Hash;
use Illuminate\Pagination\LengthAwarePaginator;

class UserService
{

    public function getAllUsersWithDetails(int $perPage = 10): LengthAwarePaginator
    {
        try {
            $users = User::with(['roles', 'staftDetail'])->paginate($perPage);
            return $users;
        } catch (Exception $e) {
            Log::error('Error al obtener la lista de usuarios con detalles: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return new LengthAwarePaginator([], 0, $perPage);
        }
    }

    public function getUserByRole($role, int $perPage = 10): LengthAwarePaginator
    {
        return User::whereHas("roles", function ($query) use ($role) {
            $query->where("name", $role);
        })->paginate($perPage);
    }

    public function getUserById($id): ?User
    {
        try {
            $user = User::find($id);
            return $user ?? null;
        } catch (Exception $e) {
            Log::error('No se pudo encontrar el usuario: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(RegisterDTO $dto): ?User
    {
        try {
            $rolFind = Role::where('name', $dto->rol_name)->first();
            if(!$rolFind){
                return null;
            }
            $user = User::create([
                'name' => $dto->name,
                'email' => $dto->email,
                'password' => Hash::make($dto->password),
            ]);
            $user->roles()->syncWithoutDetaching([$rolFind->id]);
            return $user;
        } catch (Exception $e) {
            Log::error('Error al crear usuario: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $userId, array $data): bool
    {
        try {
            $user = User::find($userId);
            if (!$user) {
                return false;
            }

            $updateData = [
                'name' => $data['name'],
                'email' => $data['email'],
                'active' => $data['active'] ?? $user->active,
            ];

            $user->update($updateData);

            if (isset($data['rol_name'])) {
                $rolFind = Role::where('name', $data['rol_name'])->first();
                if ($rolFind) {
                    $user->roles()->sync([$rolFind->id]); 
                }
            }

            return true;

        } catch (Exception $e) {
            Log::error('Error al actualizar usuario: ' . $e->getMessage(), [
                'userId' => $userId,
                'data' => $data,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }

    public function delete(int $id): bool
    {
        try {
            $user = User::find($id);
            return $user ? $user->delete() : false;
        } catch (Exception $e) {
            Log::error('Error al elimnar el usuario: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }

    public function updatePassword(int $userId, UpdatePasswordDTO $dto): bool
    {
        try {
            $user = $this->getUserById($userId);
            
            if (!$user) {
                Log::warning("Intento de actualizar contraseña para usuario inexistente.", ['userId' => $userId]);
                return false;
            }
            $user->password = Hash::make($dto->new_password);
            return $user->save();
        } catch (Exception $e) {
            Log::error('Error al actualizar la contraseña del usuario: ' . $e->getMessage(), [
                'userId' => $userId,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }
}