<?php

namespace App\Services;

use App\DTOs\StatusPipeline\StatusPipelineCreateDto;
use App\Models\StatusPipeline;
use App\Support\OperationResult;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;

class StatusPipelineService
{
    public function get(): Collection
    {
        return StatusPipeline::orderBy('order_position', 'asc')->get();
    }

    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return StatusPipeline::orderBy('order_position', 'asc')->paginate($perPage);
    }

    public function find(int $id): ?StatusPipeline
    {
        try {
            return StatusPipeline::find($id);
        } catch (Exception $e) {
            Log::error('Error al encontrar StatusPipeline: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(StatusPipelineCreateDto $statusPipelineDto): ?StatusPipeline
    {
        try {
            $maxOrder = StatusPipeline::max('order_position') ?? 0;
            $data = $statusPipelineDto->toArray();
            $data['order_position'] = $maxOrder + 1;

            return StatusPipeline::create($data);
        } catch (Exception $e) {
            Log::error('Error al crear StatusPipeline: ' . $e->getMessage(), [
                'data' => $statusPipelineDto->toArray(),
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, StatusPipelineCreateDto $statusPipelineDto): ?StatusPipeline
    {
        try {
            $statusPipeline = StatusPipeline::find($id);
            if ($statusPipeline) {
                $statusPipeline->update($statusPipelineDto->toArray());
            }
            return $statusPipeline;
        } catch (Exception $e) {
            Log::error('Error al actualizar StatusPipeline: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $statusPipelineDto->toArray(),
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): OperationResult
    {
        try {
            $statusPipeline = StatusPipeline::find($id);
            if (!$statusPipeline) {
                return OperationResult::error("El estado no existe.");
            }
            if ($statusPipeline->leads()->exists()) {
                return OperationResult::error("No se puede eliminar tiene leads asociados.");
            }
            $statusPipeline->delete();

            return OperationResult::success(message: "Estado de pipeline eliminado correctamente.");
        } catch (Exception $e) {
            Log::error('Error al eliminar StatusPipeline: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
           return OperationResult::error("Hubo un error al eliminar el estado pipeline.");
        }
    }

    public function reorder(array $items): bool
    {
        try {
            foreach ($items as $item) {
                StatusPipeline::where('id', $item['id'])
                    ->update(['order_position' => $item['order']]);
            }
            return true;
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return false;
        }
    }
}