<?php

namespace App\Services;

use App\DTOs\Staft\StaftCreateDto;
use App\DTOs\Staft\StaftDetailDto;
use App\Models\StaftDetail;
use App\Models\User;
use App\Queries\StaftListQuery;
use App\Queries\StaftPaginateListQuery;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Pagination\LengthAwarePaginator;

class StaftService
{

    public function getStaftForLeads(): Collection
    {
        return StaftListQuery::build(['superadmin'])->get();
    }

    public function get(array $excludedRoles = ['superadmin', 'admin']): Collection
    {
        $query = User::where('active', true)
            ->select('users.id', 'users.name as name')
            ->with([
                'staftDetail' => function ($query) {
                    $query->select('user_id', 'last_name')->with('documents');
                },
                'roles:id,display_name',
            ])
            ->whereDoesntHave('roles', function ($query) use ($excludedRoles) {
                $query->whereIn('name', $excludedRoles);
            });
        return $query
            ->orderBy('name', 'asc')
            ->get();
    }


    public function getUserDetailSummary(int $userId): ?StaftDetailDto
    {
        try {
            $user = User::select(['id', 'name', 'email', 'active'])
                ->with([
                    'staftDetail:user_id,department_id,phone,last_name',
                    'staftDetail.department:id,name',
                    'roles:id,name',
                    'staftDetail.documents:documentable_id,documentable_type,file_path',
                    'leadTypes:id'
                ])
                ->find($userId);

            if (!$user) {
                return null;
            }
            $user->lead_type_id = $user->leadTypes->count() > 1 ? 0 : $user->leadTypes->first()?->id;
            return StaftDetailDto::fromModel($user);

        } catch (Exception $e) {
            Log::error('Error al obtener el detalle del usuario: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'userId' => $userId
            ]);
            return null;
        }
    }

    public function getUsersSummary(array $filters = [], int $perPage = 10): LengthAwarePaginator
    {
        try {
            $usersPaginator = StaftPaginateListQuery::build($filters)->paginate($perPage);

            $usersSummary = $usersPaginator->through(function ($user) {
                $roleName = $user->roles->first()->display_name ?? 'N/A';
                $departmentName = $user->staftDetail?->department->name ?? 'N/A';
                $photoUrl = $user->staftDetail?->documents->first()->file_path ?? null;
                $lastName = $user->staftDetail?->last_name;
                $fullName = trim($user->name . ' ' . $lastName);
                return [
                    'id' => $user->id,
                    'name' => $fullName,
                    'department' => $departmentName,
                    'role' => $roleName,
                    'photo_url' => $photoUrl,
                    'active' => $user->active
                ];
            });

            return $usersSummary;

        } catch (Exception $e) {
            Log::error('Error al obtener el resumen de usuarios: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return new LengthAwarePaginator([], 0, $perPage);
        }
    }

    public function create(StaftCreateDto $dto, array $leadTypeIds = [], ?array $fileData = null): ?StaftDetail
    {
        try {
            $staftDetail = StaftDetail::create($dto->toArray());

            if (!$staftDetail) {
                throw new Exception('Fallo al crear el registro StaftDetail.');
            }
            if ($fileData) {
                $staftDetail->documents()->create([
                    'file_path' => $fileData['file_path'],
                    'file_name' => $fileData['file_name'],
                    'mime_type' => $fileData['mime_type'],
                    'file_size' => $fileData['file_size'],
                    'file_hash' => $fileData['file_hash'] ?? null,
                ]);
            }
            if (!empty($leadTypeIds)) {
                $staftDetail->user->leadTypes()->sync($leadTypeIds);
            }
            return $staftDetail;
        } catch (Exception $e) {
            Log::error('Error al los detalles del staft: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
                'file_data' => $fileData,
            ]);
            return null;
        }
    }

    public function update(int $userId, StaftCreateDto $dto, array $leadTypeIds = [], ?array $newFileData = null): ?StaftDetail
    {
        try {
            $staftDetail = StaftDetail::where('user_id', $userId)->first();

            if (!$staftDetail) {
                $dto->user_id = $userId;
            }

            $staftDetail->update($dto->toArray());
            if ($newFileData) {
                $filePath = $newFileData['file_path'] ?? null;
                $oldDocument = $staftDetail->documents()->first();
                if ($filePath !== null) {
                    $dataToUpdate = [
                        'file_path' => $filePath,
                        'file_name' => $newFileData['file_name'] ?? null,
                        'mime_type' => $newFileData['mime_type'] ?? null,
                        'file_size' => $newFileData['file_size'] ?? 0,
                        'file_hash' => $newFileData['file_hash'] ?? null,
                    ];
                    if ($oldDocument) {
                        $oldDocument->update($dataToUpdate);
                    } else {
                        $staftDetail->documents()->create($dataToUpdate);
                    }
                } else {
                    if ($oldDocument) {
                        $oldDocument->update([
                            'file_path' => null,
                            'file_name' => null,
                            'mime_type' => null,
                            'file_size' => 0,
                            'file_hash' => null,
                        ]);
                    }
                }
            }

            if (!empty($leadTypeIds)) {
                $staftDetail->user->leadTypes()->sync($leadTypeIds);
            }

            return $staftDetail;
        } catch (Exception $e) {
            Log::error('Error al actualizar los detalles del staft: ' . $e->getMessage(), [
                'userId' => $userId,
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
                'file_data' => $newFileData,
            ]);
            return null;
        }
    }

    public function haveLeadTypeAssigned(int $user_id, int $leadTypeId): bool
    {
        try {
            return User::where('id', $user_id)
                ->whereHas('staftDetail', function ($query) use ($leadTypeId) {
                    $query->where('lead_type_id', $leadTypeId);
                })
                ->exists();
        } catch (Exception $e) {
            Log::error('Error al verificar el tipo de lead asignado: ' . $e->getMessage(), [
                'userId' => $user_id,
                'leadTypeId' => $leadTypeId
            ]);
            return false;
        }
    }
}