<?php

namespace App\Services;

use App\DTOs\Job\JobCreateDTO;
use App\Models\Job;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class JobService
{
    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return Job::orderBy('name', 'asc')->paginate($perPage);
    }

    public function find(int $id): ?Job
    {
        try {
            return Job::find($id);
        } catch (Exception $e) {
            Log::error('Error al encontrar cargo: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(JobCreateDTO $jobCreateDTO): ?Job
    {
        try {
            return Job::create($jobCreateDTO->toArray());
        } catch (Exception $e) {
            Log::error('Error al crear cargo: ' . $e->getMessage(), [
                'data' => $jobCreateDTO,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, JobCreateDTO $jobCreateDTO): ?Job
    {
        try {
            $job = Job::find($id);
            if ($job) {
                $job->update($jobCreateDTO->toArray());
            }
            return $job;
        } catch (Exception $e) {
            Log::error('Error al actualizar cargo: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $jobCreateDTO,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): bool
    {
        try {
            $jobDelete = Job::find($id);
            return $jobDelete ? $jobDelete->delete() : false;
        } catch (Exception $e) {
            Log::error('Error al eliminar cargo: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }
}