<?php

namespace App\Services;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr;

class FileUploader
{
    protected string $disk = 'public';
    protected string $tempDisk = 'public'; 
    protected string $tempDirectory = 'temp/uploads';

    public function __construct(string $disk = 'public')
    {
        $this->disk = $disk;
    }

    public function storeTemp(UploadedFile $file): array
    {
        try {
            $path = $file->store($this->tempDirectory, $this->tempDisk);
            $metadata = [
                'file_name' => $file->getClientOriginalName(),
                'mime_type' => $file->getMimeType(),
                'file_size' => $file->getSize(),
                'file_hash' => hash_file('sha256', $file->getRealPath()),
                'temp_url' => Storage::disk($this->tempDisk)->url($path),
                'temp_path' => $path,
            ];
            return ['success' => true, 'data' => $metadata];

        } catch (\Exception $e) {
            Log::error('Error de subida temporal en FileUploader: ' . $e->getMessage());
            return [
                'success' => false, 
                'error' => 'No se pudo subir la foto temporalmente. Intenta con un archivo diferente.'
            ];
        }
    }

    public function moveFromTemp(array $fileData, string $finalDirectory): ?array
    {
        $tempPath = Arr::get($fileData, 'temp_path');
        if (!$tempPath) {
            return null;
        }
        
        try {
            $fileName = basename($tempPath); 
            $finalPath = $finalDirectory . '/' . $fileName;
            Storage::disk($this->disk)->move($tempPath, $finalPath);
            $fileData['file_path'] = $finalPath;
            unset($fileData['temp_path']); 
            unset($fileData['temp_url']); 
            
            return $fileData;
            
        } catch (\Exception $e) {
            Log::error("Error al mover el archivo de $tempPath a $finalDirectory: " . $e->getMessage());
            // Intenta eliminar el archivo temporal si el movimiento falló
            $this->deleteTemp($tempPath); 
            return null;
        }
    }

    public function deleteEnd(string $filePath): bool
    {
        try {
            return Storage::disk($this->disk)->delete($filePath);
        } catch (\Exception $e) {
            \Log::warning("No se pudo eliminar el archivo $filePath: " . $e->getMessage());
            return false;
        }
    }

    public function deleteTemp(string $tempPath): bool
    {
        try {
            return Storage::disk($this->tempDisk)->delete($tempPath);
        } catch (\Exception $e) {
            \Log::warning("No se pudo eliminar el archivo temporal en $tempPath: " . $e->getMessage());
            return false;
        }
    }
}