<?php

namespace App\Services;

use App\DTOs\Department\DepartmentCreateDto;
use App\Models\Department;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;

class DeparmentService
{
    public function get(): Collection
    {
        return Department::where('active', true)->orderBy('name', 'asc')->get();
    }

    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return Department::orderBy('name', 'asc')->paginate($perPage);
    }

    public function find(int $id): ?Department
    {
        try {
            return Department::find($id);
        } catch (Exception $e) {
            Log::error('Error al encontrar departamento: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(DepartmentCreateDto $departmentDto): ?Department
    {
        try {
            return Department::create($departmentDto->toArray());
        } catch (Exception $e) {
            Log::error('Error al crear departamento: ' . $e->getMessage(), [
                'data' => $departmentDto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, DepartmentCreateDto $departmentDto): ?Department
    {
        try {
            $department = Department::find($id);
            if ($department) {
                $department->update($departmentDto->toArray());
            }
            return $department;
        } catch (Exception $e) {
            Log::error('Error al actualizar departamento: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $departmentDto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): bool
    {
        try {
            $departmentDelete = Department::find($id);
            return $departmentDelete ? $departmentDelete->delete() : false;
        } catch (Exception $e) {
            Log::error('Error al eliminar departamento: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }
}