<?php

namespace App\Services;

use App\DTOs\Dashboard\FilterRangeDateDTO;
use App\Http\Resources\Lead\LeadRecentResource;
use App\Models\Lead;
use App\Models\LeadType;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class DashboardService
{
    public function getStats(FilterRangeDateDTO $filters): array
    {
        return [
            'leadsRecent' => $this->getRecentLeads($filters),
        ];
    }

    public function getTotalSales(FilterRangeDateDTO $filters): array
    {
        $cacheKey = 'total_sales_global_' . md5(serialize($filters));

        return Cache::remember($cacheKey, now()->addMinutes(30), function () use ($filters) {
            $totalSum = Lead::query()
                ->when($filters->hasDateRange(), function ($query) use ($filters) {
                    $query->whereBetween('created_at', [$filters->startDate, $filters->endDate]);
                })
                ->sum('total_cost');
            return [
                'value' => '$' . number_format($totalSum / 1000, 1) . 'k'
            ];
        });
    }

    public function getTotalLeadsCount(FilterRangeDateDTO $filters): array
    {
        $cacheKey = 'total_leads_' . md5(json_encode($filters));

        return Cache::remember($cacheKey, now()->addMinutes(value: 30), function () use ($filters) {
            $count = Lead::query()
                ->when($filters->hasDateRange(), function ($query) use ($filters) {
                    $query->whereBetween('created_at', [$filters->startDate, $filters->endDate]);
                })
                ->count();

            return ['value' => number_format($count)];
        });
    }

    private function getRecentLeads(FilterRangeDateDTO $filters): Collection
    {
        try {
            return Cache::remember(
                $filters->getCacheKey(),
                now()->addHours(1),
                function () use ($filters) {

                    $leads = Lead::query()
                        ->select(['id', 'name_event', 'lead_type_id', 'estimated_budget', 'created_at', 'slug'])
                        ->with(['leadType:id,name'])
                        ->when($filters->hasDateRange(), function ($query) use ($filters) {
                            $query->whereBetween('created_at', [$filters->startDate, $filters->endDate]);
                        })
                        ->latest()
                        ->limit($filters->limit)
                        ->get();

                    return collect(
                        LeadRecentResource::collection($leads)->resolve()
                    );
                }
            );
        } catch (\Exception $e) {
            Log::error("Dashboard Error: " . $e->getMessage());
            return collect([]);
        }
    }

    public function getLeadTypeStats(FilterRangeDateDTO $filters): array
    {
        return Cache::remember('stats_lead_types_' . $filters->getCacheKey(), now()->addMinutes(30), function () use ($filters) {
            $data = Lead::query()
                ->join('lead_types', 'leads.lead_type_id', '=', 'lead_types.id')
                ->selectRaw('lead_types.name as label, SUM(leads.total_cost) as total, lead_types.background_color')
                ->when($filters->hasDateRange(), function ($query) use ($filters) {
                    $query->whereBetween('leads.created_at', [$filters->startDate, $filters->endDate]);
                })
                ->groupBy('lead_types.name', 'lead_types.background_color')
                ->get();

            $values = $data->pluck('total')->toArray();
            $totalSum = array_sum($values);

            return [
                'labels' => $data->pluck('label')->toArray(),
                'values' => $values,
                'colors' => $data->pluck('background_color')->map(fn($c) => $c ?? '#ccc')->toArray(),
                'totalSum' => '$' . number_format($totalSum / 1000, 1) . 'k'
            ];
        });
    }

    public function getMonthlyEventStats(): array
    {
        $year = now()->year;
        $version = Cache::get('leads_cache_version', '1');
        $cacheKey = "monthly_stats_{$year}_v{$version}";

        return Cache::remember($cacheKey, now()->addHours(1), function () use ($year) {
            $leadTypes = LeadType::all();

            $results = Lead::query()
                ->selectRaw('lead_type_id, MONTH(created_at) as month, COUNT(*) as total')
                ->whereYear('created_at', $year)
                ->groupBy('lead_type_id', 'month')
                ->get();

            $monthsLabels = ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];
            $datasets = [];

            foreach ($leadTypes as $type) {
                $values = [];
                for ($m = 1; $m <= 12; $m++) {
                    $found = $results->where('lead_type_id', $type->id)->where('month', $m)->first();
                    $values[] = $found ? $found->total : 0;
                }

                $datasets[] = [
                    'label' => $type->name,
                    'values' => $values,
                    'color' => $type->background_color ?? '#8C8E90'
                ];
            }

            return [
                'labels' => $monthsLabels,
                'datasets' => $datasets
            ];
        });
    }
}