<?php

namespace App\Services;

use App\DTOs\Comment\CommentCreateDto;
use App\Models\Comment;
use App\Services\FileUploader;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class CommentService
{
    protected string $storageFolder = 'lead_comments';

    public function __construct(protected FileUploader $fileUploader)
    {}

    public function createComment(CommentCreateDto $dto, array $files = []): ?Comment
    {
        try {
            DB::beginTransaction();

            $comment = Comment::create($dto->toArray());

            if (!$comment) {
                throw new Exception('Fallo al crear el registro del Comentario.');
            }
            if (!empty($files)) {
                $attachmentsData = $this->handleAttachments($files);

                foreach ($attachmentsData as $fileData) {
                    $finalFileData = $this->fileUploader->moveFromTemp($fileData, $this->storageFolder);

                    if (!$finalFileData) {
                        throw new Exception('Ocurrió un error al mover un archivo a su ubicación final.');
                    }
                    
                    $comment->documents()->create([
                        'file_path' => $finalFileData['file_path'],
                        'file_name' => $finalFileData['file_name'],
                        'mime_type' => $finalFileData['mime_type'],
                        'file_size' => $finalFileData['file_size'],
                        'file_hash' => $finalFileData['file_hash'] ?? null,
                    ]);
                }
            }

            DB::commit();
            return $comment;

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error al crear comentario con adjuntos: ' . $e->getMessage(), [
                'data' => $dto,
                'files' => $files,
                'trace' => $e->getTraceAsString(),
            ]);

            if (isset($attachmentsData)) {
                foreach ($attachmentsData as $fileData) {
                    if (isset($fileData['temp_path'])) {
                         $this->fileUploader->deleteTemp($fileData['temp_path']);
                    }
                }
            }
            throw $e;
        }
    }
    
    /**
     * Sube temporalmente los archivos antes de la transacción.
     * @param array<UploadedFile> $files
     * @return array<array>
     */
    protected function handleAttachments(array $files): array
    {
        $uploadedFilesData = [];
        
        foreach ($files as $file) {
            if ($file instanceof UploadedFile) {
                $result = $this->fileUploader->storeTemp($file);
                
                if ($result['success']) {
                    $uploadedFilesData[] = $result['data'];
                } else {
                    throw new Exception("Error en la subida temporal de un archivo: {$result['error']}");
                }
            }
        }
        
        return $uploadedFilesData;
    }
}