<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Str;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Lead extends Model
{
    use HasFactory;
    use LogsActivity;

    protected static $logAttributes = [
        'name_event',
        'name',
        'email',
        'phone',
        'country',
        'number_of_people',
        'estimated_budget',
        'lead_type_id',
        'status_pipeline_id',
        'venue_id',
        'ability_max',
        'rooms_guests',
        'money_budget',
        'last_name',
        'additional_information'
    ];

    private static $attributeNames = [
        'name_event' => 'Nombre del Evento',
        'name' => 'Nombre',
        'email' => 'Correo Electrónico',
        'phone' => 'Teléfono',
        'country' => 'País',
        'number_of_people' => 'Número de Personas',
        'estimated_budget' => 'Presupuesto Estimado',
        'lead_type_id' => 'Tipo de Lead',
        'status_pipeline_id' => 'Estado del Pipeline',
        'venue_id' => 'Lugar/Ubicación',
        'ability_max' => 'Capacidad Máxima',
        'rooms_guests' => 'Habitaciones/Huéspedes',
        'money_budget' => 'Presupuesto Monetario',
        'last_name' => 'Apellido',
        'additional_information' => 'Información Adicional',
        'checkin_at' => 'Fecha de llegada',
        'checkout_at' => 'Fecha de salida',
        'is_flexible' => 'Fechas flexibles',
        'flexible_month' => 'Mes - Fechas Flexible',
        'flexible_duration' => 'Número de noches - Fechas Flexible',
        'total_cost' => 'Costo total'
    ];

    protected $fillable = [
        'name_event',
        'name',
        'email',
        'phone',
        'country',
        'number_of_people',
        'estimated_budget',
        'lead_type_id',
        'status_pipeline_id',
        'venue_id',
        'ability_max',
        'rooms_guests',
        'money_budget',
        'last_name',
        'additional_information',
        'checkin_at',
        'checkout_at',
        'is_flexible',
        'flexible_month',
        'flexible_duration',
        'slug',
        'total_cost',
    ];

    public static function getAttributeName(string $key): string
    {
        return self::$attributeNames[$key] ?? ucfirst(str_replace('_', ' ', $key));
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logFillable()
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function assignedUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'lead_user')
            ->withPivot('assignment_method')
            ->withTimestamps();
    }

    public function documents(): MorphMany
    {
        return $this->morphMany(Document::class, 'documentable');
    }

    public function leadType()
    {
        return $this->belongsTo(LeadType::class, 'lead_type_id');
    }

    public function statusPipeline()
    {
        return $this->belongsTo(StatusPipeline::class, 'status_pipeline_id');
    }

    public function venue()
    {
        return $this->belongsTo(Venue::class, 'venue_id');
    }

    public function comments(): HasMany
    {
        return $this->hasMany(Comment::class);
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($lead) {
            $slug = Str::slug($lead->name_event);
            $originalSlug = $slug;
            $count = 1;

            while (static::where('slug', $slug)->exists()) {
                $slug = $originalSlug . '-' . $count;
                $count++;
            }

            $lead->slug = $slug;
        });
    }
}