<?php

namespace App\Http\Requests\Staft;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class StaftCreateRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8'],
            'last_name' => ['required', 'string', 'max:255'],
            'rol_name' => [
                'required',
                'string',
                Rule::exists('roles', 'name'),
            ],
            'photo_file' => [
                'nullable',
                'file',
                'image',
                'max:5048',
                'mimes:jpg,jpeg,png,webp',
            ],
            'phone' => ['nullable', 'string', 'regex:/^\+(?:[0-9] ?){6,16}[0-9]$/'],
            'department_id' => [
                'nullable',
                'string',
                Rule::exists('departments', 'id'),
            ],
            'lead_type_id' => [
                'required',
                function ($attribute, $value, $fail) {
                    if ($value === "0" || $value === 0) {
                        return;
                    }
                    $exists = DB::table('lead_types')
                        ->where('id', $value)
                        ->exists();

                    if (!$exists) {
                        $fail('El tipo de evento seleccionado no es válido.');
                    }
                },
            ],
            'active' => [
                'nullable',
                'boolean',
            ],
        ];
    }

    public function messages(): array
    {
        return [
            'name.required' => 'El nombre es obligatorio.',
            'name.string' => 'El nombre debe ser una cadena de texto.',
            'name.max' => 'El nombre no puede exceder los :max caracteres.',

            'last_name.required' => 'El apellido es obligatorio.',
            'last_name.string' => 'El apellido debe ser una cadena de texto.',
            'last_name.max' => 'El apellido no puede exceder los :max caracteres.',

            'email.required' => 'El correo electrónico es obligatorio.',
            'email.email' => 'El correo electrónico debe tener un formato válido.',
            'email.unique' => 'Este correo electrónico ya está registrado.',
            'email.max' => 'El correo electrónico no puede exceder los :max caracteres.',

            'password.required' => 'La contraseña es obligatoria.',
            'password.min' => 'La contraseña debe tener al menos :min caracteres.',

            'rol_name.required' => 'El rol del usuario es obligatorio.',
            'rol_name.exists' => 'El rol seleccionado no es válido.',

            'photo_file.file' => 'La foto debe ser un archivo válido.',
            'photo_file.image' => 'El archivo subido debe ser una imagen.',
            'photo_file.max' => 'El tamaño máximo de la imagen es de :max kilobytes (2MB).',
            'photo_file.mimes' => 'La imagen debe ser de tipo: jpg, jpeg, png o webp.',

            'phone.max' => 'El número de teléfono no puede exceder los :max caracteres.',
            'phone.string' => 'El número de teléfono debe ser una cadena de texto.',

            'deparment_id.exists' => 'El departamento seleccionado no es válido.',

            'lead_type_id.exists' => 'El tipo de lead seleccionado no es válido.',
            'lead_type_id.required' => 'El tipo de evento es obligatorio.',

            'active.boolean' => 'El campo "activo" debe ser verdadero o falso.',
        ];
    }
    protected function prepareForValidation(): void
    {
        if ($this->has('active')) {
            $this->merge([
                'active' => filter_var($this->active, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? $this->active,
            ]);
        }
    }
}
