<?php

namespace App\Http\Requests\Lead;

use Illuminate\Foundation\Http\FormRequest;

class CreateLeadApiRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'lead_type_id' => ['required', 'integer', 'exists:lead_types,id'],
            'name_event' => ['required', 'string', 'max:255'],
            'venue_id' => ['required', 'integer', 'exists:venues,id'],
            'number_of_people' => ['nullable', 'integer', 'min:1'],
            'rooms_guests' => ['nullable', 'integer', 'min:1'],
            'estimated_budget' => ['required', 'numeric', 'min:0'],
            'name' => ['required', 'string', 'max:100'],
            'last_name' => ['required', 'string', 'max:100'],
            'email' => ['required', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
            'country' => ['nullable', 'string', 'max:255'],
            'additional_information' => ['nullable', 'string'],

            'is_flexible' => ['required', 'boolean'],
            'flexible_month' => ['required_if:is_flexible,true', 'nullable', 'string'],
            'flexible_duration' => ['required_if:is_flexible,true', 'nullable', 'integer', 'min:1'],

            'checkin_at' => ['required_if:is_flexible,false', 'nullable', 'date'],
            'checkout_at' => ['required_if:is_flexible,false', 'nullable', 'date', 'after:checkin_at'],
        ];
    }

    public function messages(): array
    {
        return [
            'lead_type_id.required' => 'El tipo de evento es obligatorio.',
            'lead_type_id.exists' => 'El tipo de evento seleccionado no es válido.',

            'name_event.required' => 'El nombre del evento es obligatorio.',
            'name_event.max' => 'El nombre del evento no puede superar los :max caracteres.',

            'venue_id.exists' => 'El Venue seleccionado no es válido.',

            'number_of_people.min' => 'El número de asistentes debe ser al menos :min.',
            'number_of_people.lte' => 'El número de asistentes no puede exceder la capacidad máxima.',

            'rooms_guests.min' => 'El número de habitaciones debe ser al menos :min.',

            'estimated_budget.required' => 'El presupuesto estimado es obligatorio.',
            'estimated_budget.numeric' => 'El presupuesto debe ser un valor numérico.',
            'estimated_budget.min' => 'El presupuesto debe ser un valor positivo o cero.',

            'name.required' => 'El nombre del contacto es obligatorio.',
            'name.max' => 'El nombre no puede superar los :max caracteres.',

            'last_name.required' => 'El apellido del contacto es obligatorio.',
            'last_name.max' => 'El apellido no puede superar los :max caracteres.',

            'email.required' => 'El correo electrónico es obligatorio.',
            'email.email' => 'El formato del correo electrónico no es válido.',
            'email.max' => 'El correo electrónico no puede superar los :max caracteres.',

            'phone.max' => 'El número de teléfono no puede superar los :max caracteres.',
            'country.max' => 'El país no puede superar los :max caracteres.',
            'additional_information.string' => 'La información adicional debe ser texto.',

            'is_flexible.boolean' => 'El campo de fechas flexibles no es válido.',

            'checkin_at.required_if' => 'La fecha de llegada es obligatoria.',
            'checkin_at.date' => 'La fecha de llegada debe ser una fecha válida.',

            'checkout_at.required_if' => 'La fecha de salida es obligatoria.',
            'checkout_at.date' => 'La fecha de salida debe ser una fecha válida.',
            'checkout_at.after' => 'La fecha de salida debe ser posterior a la fecha de llegada.',

            'flexible_month.required_if' => 'El mes es obligatorio.',
            'flexible_month.string' => 'El mes flexible debe ser un texto válido.',

            'flexible_duration.required_if' => 'Número de noches es obligatorio.',
            'flexible_duration.integer' => 'La duración flexible debe ser un número entero.',
            'flexible_duration.min' => 'La duración flexible debe ser de al menos :min día.',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'is_flexible' => filter_var($this->is_flexible, FILTER_VALIDATE_BOOLEAN),
        ]);
    }
}