<?php

namespace App\Http\Controllers\Lead;

use App\DTOs\Comment\CommentCreateDto;
use App\DTOs\Lead\LeadCreateDTO;
use App\DTOs\Lead\LeadUpdateDTO;
use App\Enums\AssignmentMethod;
use App\Http\Controllers\Controller;
use App\Http\Requests\Comment\CommentCreateRequest;
use App\Http\Requests\Lead\CreateLeadRequest;
use App\Http\Requests\Lead\UpdateLeadRequest;
use App\Services\CommentService;
use App\Services\DocumentService;
use App\Services\LeadService;
use App\Services\LeadTypeService;
use App\Services\StaftService;
use App\Services\StatusPipelineService;
use App\Services\VenueService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class LeadController extends Controller
{
    public function __construct(
        protected LeadService $leadService,
        protected StatusPipelineService $statusPipelineService,
        protected LeadTypeService $leadTypeService,
        protected StaftService $staftService,
        protected VenueService $venueService,
        protected CommentService $commentService,
        protected DocumentService $documentService
    ) {
    }

    public function index(Request $request): View
    {
        $filters = [
            'type_event'  => $request->integer('type_event'),
            'state_event' => $request->integer('state_event'),
            'venue_id'    => $request->integer('venue_id'),
        ];

        $listLeads = $this->leadService->getPaginated(auth()->user(), array_filter($filters));
        $leadTypes = $this->leadTypeService->get();
        $typesPipelines = $this->statusPipelineService->get();
        $venues = $this->venueService->get();
        return view('leads.index', compact('listLeads', 'leadTypes', 'typesPipelines', 'venues'));
    }

    public function create(): View
    {
        $typesPipelines = $this->statusPipelineService->get();
        $staftList = $this->staftService->getStaftForLeads();
        $leadTypes = $this->leadTypeService->get();
        $venues = $this->venueService->get();
        return view('leads.create', compact('typesPipelines', 'staftList', 'leadTypes', 'venues'));
    }

    public function store(CreateLeadRequest $createLeadRequest): RedirectResponse
    {
        $validatedData = $createLeadRequest->validated();
        $currentUser = auth()->user();
        $syncData = [];

        if ($currentUser->hasAnyRole(['admin', 'superadmin'])) {
            $userIds = $createLeadRequest->input('users_assigned_ids', []);
            foreach ($userIds as $id) {
                $syncData[$id] = ['assignment_method' => AssignmentMethod::MANUAL->value];
            }
        } else {
            $syncData[$currentUser->id] = ['assignment_method' => AssignmentMethod::AUTOMATIC->value];
        }
        $dto = new LeadCreateDTO($validatedData);
        $leadNew = $this->leadService->create($dto, $syncData);

        if (!$leadNew) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Ocurrió un error al guardar el lead. Intenta nuevamente.']);
        }

        return redirect()->route('leads.index')->with('success', 'Lead creado correctamente.');
    }

    public function edit($slug): View
    {
        $perPage = 10;
        $detailLead = $this->leadService->find($slug);
        if (!$detailLead) {
            abort('404');
        }
        $typesPipelines = $this->statusPipelineService->get();
        $staftList = $this->staftService->getStaftForLeads();
        $leadTypes = $this->leadTypeService->get();
        $venues = $this->venueService->get();

        $selectedIds = $detailLead->assignedUsers->pluck('id')->toArray();
        $detailTime = $this->leadService->getTimeline($detailLead, $perPage);
        $detailCommets = $this->leadService->getCommentsWithDocuments($detailLead, $perPage);
        $detailDocuments = $this->documentService->getCommentDocumentsForLead($detailLead);
        return view('leads.edit', compact('detailLead', 'typesPipelines', 'staftList', 'leadTypes', 'venues', 'detailLead', 'detailTime', 'detailCommets', 'detailDocuments', 'perPage', 'selectedIds'));
    }

    public function update(UpdateLeadRequest $updateLeadRequest, $id): RedirectResponse
    {
        $dto = new LeadUpdateDTO($updateLeadRequest->validated());
        $currentUser = auth()->user();
        $syncData = [];
        if ($currentUser->hasAnyRole(['admin', 'superadmin'])) {
            $userIds = $updateLeadRequest->input('users_assigned_ids', []);
            foreach ($userIds as $userId) {
                $syncData[$userId] = ['assignment_method' => AssignmentMethod::MANUAL->value];
            }
        }
        $leadUpdate = $this->leadService->update($id, $dto, $syncData);
        if (!$leadUpdate) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'No se pudo actualizar el lead.']);
        }
        return redirect()->route('leads.index')->with('success', 'Lead actualizado correctamente.');
    }

    public function storeComment(CommentCreateRequest $request, int $leadId) : JsonResponse
    {
        $validatedData = $request->validated();
        $files = $request->file('attachments') ?? [];
        $user = auth()->user();
        try {
            $dto = new CommentCreateDto([
                'body' => $validatedData['body'],
                'lead_id' => $leadId,
                'user_id' => $user->id
            ]);
            $this->commentService->createComment($dto, $files);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error interno al guardar el comentario. Por favor, intenta de nuevo.',
                'error' => $e->getMessage() . ' Intenta nuevamente.'
            ], 500);
        }
        return response()->json(['message' => 'Comentario agregado correctamente.', 'redirect_url' => route('leads.edit', $leadId)], 201);
    }

    public function getTimelineAjax(Request $request, string $lead_id): JsonResponse
    {

        $leadDetail = $this->leadService->find($lead_id);
        $lastActivityId = $request->input('last_id');
        $order = $request->input('order', 'desc');
        $perPage = $request->input('per_page', 10);

        $timeline = $this->leadService->getTimeline($leadDetail, $perPage, $lastActivityId, $order);
        return response()->json([
            'timeline' => $timeline,
            'has_more' => $timeline->count() == $perPage
        ]);
    }
}