<?php

namespace App\Http\Controllers\Admin;

use App\DTOs\Auth\RegisterDTO;
use App\DTOs\Staft\StaftCreateDto;
use App\Http\Controllers\Controller;
use App\Http\Requests\Staft\StaftCreateRequest;
use App\Http\Requests\Staft\StaftUpdateRequest;
use App\Services\DeparmentService;
use App\Services\FileUploader;
use App\Services\LeadService;
use App\Services\LeadTypeService;
use App\Services\RolService;
use App\Services\StaftService;
use App\Services\UserService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;

class StaftController extends Controller
{
    public function __construct(
        protected StaftService $staftService,
        protected UserService $userService,
        protected LeadTypeService $leadTypeService,
        protected DeparmentService $deparmentService,
        protected RolService $rolService,
        protected FileUploader $fileUploader,
        protected LeadService $leadService
    ) {

    }

    public function index(Request $request): View
    {
        $filters = [
            'type_event'  => $request->integer('type_event'),
            'state_active' => $request->string('state_active')->trim()->toString(),
        ];
        // dd($filters);
        $leadTypes = $this->leadTypeService->get();
        $listData = $this->staftService->getUsersSummary($filters);
        return view('admin.stafts.index', compact('listData', 'leadTypes'));
    }

    public function create(): View
    {
        $leadTypes = $this->leadTypeService->get();
        $allLeadTypes = new Collection();
        $allLeadTypes->push((object) ['id' => 0, 'name' => 'Ambos']);
        $leadTypes = $allLeadTypes->merge($leadTypes);
        $departments = $this->deparmentService->get();
        $rols = $this->rolService->get();
        return view('admin.stafts.create', compact('leadTypes', 'departments', 'rols'));
    }

    public function edit($id): View | RedirectResponse
    {
        $leadTypes = $this->leadTypeService->get();
        $allLeadTypes = new Collection();
        $allLeadTypes->push((object) ['id' => 0, 'name' => 'Ambos']);
        $leadTypes = $allLeadTypes->merge($leadTypes);
        $departments = $this->deparmentService->get();
        $rols = $this->rolService->get();
        $detailStaft = $this->staftService->getUserDetailSummary($id);
        if(!$detailStaft){
            return redirect()->route('admin.stafts.index')->with('error', 'Lo sentimos tenemos no podemos consultar el detalle del staft.');
        }
        $listEventsAsign = $this->leadService->getLeadsByUserPaginated($detailStaft->id);
        // dd($listEventsAsign);
        return view('admin.stafts.edit', compact('detailStaft', 'leadTypes', 'departments', 'rols', 'listEventsAsign'));
    }

    public function store(StaftCreateRequest $staftCreateRequest): RedirectResponse
    {
        $validatedData = $staftCreateRequest->validated();
        $fileData = null;
        $result = null;

        if ($staftCreateRequest->hasFile('photo_file')) {
            $result = $this->fileUploader->storeTemp($staftCreateRequest->file('photo_file'));
            if ($result['success']) {
                $fileData = $result['data'];
                $validatedData['photo_file_temp_path'] = $fileData['temp_path'];
            } else {
                return redirect()->back()
                    ->withInput($validatedData)
                    ->withErrors(['photo_file' => $result['error']]);
            }
        }
        try {
            DB::beginTransaction();
            $dtoUser = new RegisterDTO($staftCreateRequest->validated());
            $userNew = $this->userService->create($dtoUser);

            if (!$userNew) {
                throw new \Exception('Ocurrió un error al crear accesos del staff.');
            }

            $finalFileData = null;

            if ($fileData) {
                $finalFileData = $this->fileUploader->moveFromTemp($fileData, 'staff_photos');
                if (!$finalFileData) {
                    throw new \Exception('Ocurrió un error al mover la foto a su ubicación final.');
                }
            }

            $selectedLeadTypeId = $validatedData['lead_type_id'];
            $selectedLeads = array();
            if ((int)$selectedLeadTypeId === 0) {
                $allRealLeadTypes = $this->leadTypeService->get();
                $selectedLeads = $allRealLeadTypes->pluck('id')->toArray();
            }else{
                $selectedLeads = [(int)$selectedLeadTypeId];
            }
            $dtoStaft = new StaftCreateDto($validatedData);
            $dtoStaft->user_id = $userNew->id;

            $detailStaftNew = $this->staftService->create($dtoStaft, $selectedLeads, $finalFileData);

            if (!$detailStaftNew) {
                throw new \Exception('Ocurrió un error al guardar la información personal del staff.');
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            if ($finalFileData) {
                $this->fileUploader->deleteEnd($finalFileData['file_path']);
            }
            if (isset($validatedData['photo_file_temp_path'])) {
                $this->fileUploader->deleteTemp($validatedData['photo_file_temp_path']);
            }
            return redirect()->back()
                ->withInput($validatedData)
                ->withErrors(['error' => $e->getMessage() . ' Intenta nuevamente.']);
        }

        return redirect()->route('admin.stafts.index')->with('success', 'Staft creado correctamente.');
    }

    public function update(StaftUpdateRequest $staftUpdateRequest, $id): RedirectResponse
    {
        $validatedData = $staftUpdateRequest->validated();
        $fileData = null;
        $finalFileData = null;
        $result = null;
        $oldFilePath = $validatedData['old_photo_file_path'] ?? null;
        $shouldDeletePhoto = ($validatedData['delete_photo'] ?? '0') === '1';
        if ($staftUpdateRequest->hasFile('photo_file')) {
            $result = $this->fileUploader->storeTemp($staftUpdateRequest->file('photo_file'));
            if ($result['success']) {
                $fileData = $result['data'];
                $validatedData['photo_file_temp_path'] = $fileData['temp_path'];
                $shouldDeletePhoto = false;
            } else {
                return redirect()->back()
                    ->withInput($validatedData)
                    ->withErrors(['photo_file' => $result['error']]);
            }
        }

        try {
            DB::beginTransaction();
            $userUpdated = $this->userService->update((int) $id, $validatedData);
            if (!$userUpdated) {
                throw new \Exception('Ocurrió un error al actualizar los accesos del staff.');
            }
            if ($fileData) {
                $finalFileData = $this->fileUploader->moveFromTemp($fileData, 'staff_photos');
                if (!$finalFileData) {
                    throw new \Exception('Ocurrió un error al mover la nueva foto a su ubicación final.');
                }
                if ($oldFilePath) {
                    $this->fileUploader->deleteEnd($oldFilePath);
                }
            } elseif ($shouldDeletePhoto) {
                if ($oldFilePath) {
                    $this->fileUploader->deleteEnd($oldFilePath);
                }
                $finalFileData = ['file_path' => null];
            }

            $dtoStaft = new StaftCreateDto($validatedData);
            $dtoStaft->user_id = $id;

            $selectedLeadTypeId = $validatedData['lead_type_id'];
            $selectedLeads = array();
            if ((int)$selectedLeadTypeId === 0) {
                $allRealLeadTypes = $this->leadTypeService->get();
                $selectedLeads = $allRealLeadTypes->pluck('id')->toArray();
            }else{
                $selectedLeads = [(int)$selectedLeadTypeId];
            }

            $detailStaftUpdated = $this->staftService->update((int) $id, $dtoStaft, $selectedLeads, $finalFileData);

            if (!$detailStaftUpdated) {
                throw new \Exception('Ocurrió un error al actualizar la información personal del staff.');
            }

            DB::commit();

        } catch (\Exception $e) {
            DB::rollBack();
            if (
                isset($finalFileData) && is_array($finalFileData) && isset($finalFileData['file_path']) &&
                is_string($finalFileData['file_path'])
            ) {
                $this->fileUploader->deleteEnd($finalFileData['file_path']);
            }
            if (isset($validatedData['photo_file_temp_path'])) {
                $this->fileUploader->deleteTemp($validatedData['photo_file_temp_path']);
            }
            return redirect()->back()
                ->withInput($validatedData)
                ->withErrors(['error' => $e->getMessage() . ' Intenta nuevamente.']);
        }

        return redirect()->route('admin.stafts.index')->with('success', 'Staft actualizado correctamente.');
    }
}
