document.addEventListener('DOMContentLoaded', () => {
    // Array para almacenar todos los códigos generados
    let barcodesQueue = [];
    let currentStep = 1;

    // --- Selectores DOM ---
    // Usamos selectores que asumen que los elementos con estos IDs existen en tu vista home.blade.php
    const step1 = document.getElementById('step-1');
    const step2 = document.getElementById('step-2');
    const step3 = document.getElementById('step-3');
    const description = document.getElementById('step-description');

    const quantityForm = document.getElementById('quantity-form');
    const quantityInput = document.getElementById('quantity-input');
    const barcodeList = document.getElementById('barcode-list');
    const countDisplay2 = document.getElementById('count-display-2');

    const loadingIndicator = document.getElementById('loading-indicator');
    const loadingProgress = document.getElementById('loading-progress');
    const errorContainer = document.getElementById('error-container');
    const errorMessage = document.getElementById('error-message');

    const downloadWordBtn = document.getElementById('download-word-btn');
    const clearBtn = document.getElementById('clear-btn');

    // Inicialización
    // Llamar a updateStep(1) asegura que el generador inicie en el primer paso visible.
    updateStep(1); 

    // --- Funciones de Utilidad ---

    /** Muestra un error en la interfaz. */
    function showError(message) {
        errorMessage.textContent = message;
        errorContainer.classList.remove('hidden');
    }

    /** Oculta el error. */
    function hideError() {
        errorContainer.classList.add('hidden');
        errorMessage.textContent = '';
    }

    /** Muestra u oculta el indicador de carga. */
    function setLoading(show, message = '') {
        if (show) {
            loadingIndicator.classList.remove('hidden');
            loadingProgress.textContent = message;
        } else {
            loadingIndicator.classList.add('hidden');
        }
    }

    /** Actualiza la visibilidad de los pasos. */
    function updateStep(newStep) {
        currentStep = newStep;
        step1?.classList.add('hidden'); // Uso el operador '?' para seguridad en caso de que el elemento no exista
        step2?.classList.add('hidden');
        step3?.classList.add('hidden');

        if (newStep === 1) {
            step1?.classList.remove('hidden');
        } else if (newStep === 2 || newStep === 3) {
            step2?.classList.remove('hidden');
            step3?.classList.remove('hidden');
            description.textContent = 'Los códigos han sido generados. Ahora puede descargar el documento para su impresión.';
            renderBarcodes();
        }
    }

    /** Renderiza los códigos de barras en la lista. */
    function renderBarcodes() {
        barcodeList.innerHTML = '';

        barcodesQueue.forEach(item => {
            const div = document.createElement('div');
            // Se eliminan clases relacionadas con la impresión.
            div.className = 'barcode-item flex flex-col items-center justify-center p-2 m-1 border border-gray-200 rounded-sm shadow-sm';
            div.style.width = '30%'; // Permite 3 códigos por fila en pantalla

            div.innerHTML = `
                <img src="data:image/png;base64,${item.barcode_base64}" alt="Código de Barras ${item.serial_code}" class="w-full h-auto max-w-[100px] mb-1">
                <p class="text-xs font-mono text-gray-800">${item.serial_code}</p>
            `;
            barcodeList.appendChild(div);
        });

        countDisplay2.textContent = barcodesQueue.length;
    }

    /** Limpia la cola y vuelve al Paso 1. */
    function clearQueue() {
        barcodesQueue = [];
        barcodeList.innerHTML = '';
        hideError();
        setLoading(false);
        updateStep(1);
    }

    // --- Lógica de Comunicación con el Backend ---

    /** Llama al backend para generar un único código de barras. */
    async function fetchSingleBarcode() {
        const endpoint = '/api/barcode/generate';

        const response = await fetch(endpoint, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                // Asegúrate de que el token CSRF esté en el meta tag del HTML
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        if (!response.ok) {
            let errorDetails = 'Error desconocido';
            try {
                const error = await response.json();
                errorDetails = error.message || JSON.stringify(error);
            } catch (e) {
                errorDetails = `Error: ${response.status} ${response.statusText}`;
            }
            throw new Error(errorDetails);
        }

        const result = await response.json();
        return result.data;
    }

    /** Genera N códigos de barras. */
    async function generateBarcodes(quantity) {
        hideError();
        barcodesQueue = [];
        setLoading(true);

        for (let i = 0; i < quantity; i++) {
            try {
                loadingProgress.textContent = `Generando ${i + 1} de ${quantity}...`;
                const data = await fetchSingleBarcode();
                barcodesQueue.push(data);
            } catch (e) {
                showError(`Fallo al generar código ${i + 1}: ${e.message}`);
                setLoading(false);
                return;
            }
        }

        setLoading(false);
        updateStep(3); // Mover al Paso 3 (Visualización + Acción)
    }

    /** Envía los códigos en cola al endpoint para descargar el DOCX. */
    async function downloadWordDocument() {
        hideError();
        if (barcodesQueue.length === 0) {
            showError("No hay códigos para descargar.");
            return;
        }

        setLoading(true, "Preparando documento Word...");

        try {
            const response = await fetch('/api/barcode/download-word', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify({ codes: barcodesQueue })
            });

            if (response.ok) {
                const blob = await response.blob();
                let filename = 'codigos-barras.docx';
                const contentDisposition = response.headers.get('Content-Disposition');
                if (contentDisposition) {
                    const match = contentDisposition.match(/filename="?(.+)"?/);
                    if (match && match[1]) {
                        filename = match[1];
                    }
                }

                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.style.display = 'none';
                a.href = url;
                a.download = filename;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                
                description.textContent = 'Documento DOCX descargado con éxito. Ábralo en Word para imprimir.';

            } else {
                const errorText = await response.text();
                showError(`Error en el servidor al generar DOCX. Status: ${response.status}. Por favor, revise los logs del servidor.`);
                console.error("Respuesta de error del servidor:", errorText);
            }
        } catch (e) {
            showError(`Error de red al descargar DOCX: ${e.message}`);
        } finally {
            setLoading(false);
        }
    }

    // --- Event Listeners ---

    // Verificar si los elementos existen antes de agregar listeners para evitar errores
    if (quantityForm) {
        quantityForm.addEventListener('submit', function (e) {
            e.preventDefault();
            const quantity = parseInt(quantityInput.value);
            if (quantity > 0) {
                generateBarcodes(quantity);
            } else {
                showError("Por favor, ingrese un número mayor a 0.");
            }
        });
    }

    if (downloadWordBtn) {
        downloadWordBtn.addEventListener('click', downloadWordDocument);
    }
    
    if (clearBtn) {
        clearBtn.addEventListener('click', clearQueue);
    }
});